//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.data.Core 1.0

GridView {
    id : gridView

    property alias scrollAnimation : scrollAnimation

    // -------------------------------------------------------------------------
    //! Selection properties
    property var    selection      :  []
    //! previously selected index, used as anchor for multiselection
    property int    previousIndex  : currentIndex
    property bool   multiSelection : true

    // -------------------------------------------------------------------------
    //! Key Handling Signals
    signal deletePressed()
    signal enterPressed()
    signal escapePressed()
    signal leftPressed()
    signal rightPressed()
    signal upPressed()
    signal downPressed()
    signal spacePressed()

    readonly property int cellsPerRow : Math.floor(((gridView.width)
                                                    /gridView.cellWidth))

    delegate : SelectionGridDelegate {}

    onFlowChanged : {
        if (flow !== GridView.FlowLeftToRight) {
            throw new Error("SelectionGridView does " +
                            "only support left to right flow");
        }
    }

    Connections {
        target  : gridView.model
        onDocumentsChanged : {
            //if (gridView.model.isModifyable)
                p_.updateIndices();
        }
        ignoreUnknownSignals : true
    }

    QtObject {
        id : p_

        // update indices after drag&drop operation
        function updateIndices() {
            var currentDocument = selection[selection.length - 1];
            var newCurrentIndex = model.indexOf(currentDocument);

            currentIndex  = newCurrentIndex;
            previousIndex = model.indexOf(selection[0]);

        }
    }

    function selectDocumentAt(index) {
        if (index < 0 || index > model.documents.length)
            return;

        currentIndex  = index;
        previousIndex = index;
        selection     = selectRange(currentIndex, previousIndex);
    }

    function selectDocuments(documents) {
        selection     = documents;
        currentIndex  = model.indexOf(documents[0]);
        previousIndex = model.indexOf(documents[documents.length - 1]);
    }

    function removeDocument(doc) {
        gridView.model.removeElement(doc);
    }

    function selectDocument(select) {
        currentIndex = model.indexOf(select);

        if (currentIndex === -1)
            deselect();
        else {
            selection = [select];
            previousIndex = currentIndex;
        }

    }

    //! range selection function
    function selectRange(from, to, reselect) {
        if (from === -1 || to === -1)
            return [];

        if (from > to) {
            var tmp = from;
            from = to;
            to = tmp;
        }

        var docs = gridView.model.documents;
        if (docs.length === 0)
            return [];

        if (to >= docs.length)
            to = docs.length-1;

        reselect = typeof reselect !== 'undefined' ? reselect : true;

        var newSelection = [];
        for (var i = from; i <= to; ++i) {
            if (reselect || gridView.selection.indexOf(docs[i]) === -1)
                newSelection.push(docs[i]);
        }
        return newSelection;
    }

    //! Key Handling
    function initSelection(event) {
        gridView.currentIndex = gridView.model.documents.length > 0 ? 0 : -1;
        gridView.previousIndex = gridView.currentIndex;
        gridView.selection = [gridView.model.documents[currentIndex]];
        event.accepted = true;
    }

    function deselect() {
        gridView.currentIndex = -1;
        gridView.previousIndex = -1;
        gridView.selection = [];
    }

    move : Transition {
        SmoothedAnimation {
            properties : "x,y"
            duration: 200
        }
        NumberAnimation {
            duration : 150
            property : "opacity"
            to : 1
        }
    }
    displaced : move
    remove : Transition {
        NumberAnimation {
            duration : 150
            properties : "opacity,scale"
            from : 1
            to : 0.01
        }
    }
    add : Transition {
        SequentialAnimation {
            PropertyAction {
                property : "opacity"
                value : 0
            }
            PauseAnimation { duration : 100 }
            NumberAnimation {
                duration : 150
                property : "opacity"
                from : 0
                to : 1
            }
            alwaysRunToEnd : true
        }
    }
    populate : add

    NumberAnimation {
        id : scrollAnimation
        target : gridView
        property : "contentY"
        easing.type : Easing.Linear
    }

    focus : true
    keyNavigationWraps : true
    Keys.onPressed : {
        var docs = gridView.model.documents;
        var shiftMultiSelection = (event.modifiers & Qt.ShiftModifier)
                && gridView.multiSelection;

        if (event.matches(StandardKey.SelectAll) && gridView.multiSelection) {
            gridView.previousIndex = 0;//docs.length > 0 ? 0 : -1;
            gridView.currentIndex = docs.length-1;
            gridView.selection = gridView.selectRange(0, (docs.length-1));
            event.accepted = true;
            return;
        }


        if (event.key === Qt.Key_Up && shiftMultiSelection) {
            if(gridView.selection.length === 0) {
                upPressed();
                initSelection(event);
                return;
            }

            if (gridView.previousIndex === -1)
                gridView.previousIndex = gridView.currentIndex;

            gridView.moveCurrentIndexUp();
            gridView.selection = gridView.selectRange(gridView.previousIndex,
                                                      gridView.currentIndex);
            upPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Down && shiftMultiSelection) {
            if(gridView.selection.length === 0) {
                downPressed();
                initSelection(event);
                return;
            }

            if (gridView.previousIndex === -1)
                gridView.previousIndex = gridView.currentIndex;

            gridView.moveCurrentIndexDown();
            gridView.selection = gridView.selectRange(gridView.previousIndex,
                                                      gridView.currentIndex);
            downPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Left && shiftMultiSelection) {

            if(gridView.selection.length === 0) {
                leftPressed();
                initSelection(event);
                return;
            }

            if (gridView.previousIndex === -1)
                gridView.previousIndex = gridView.currentIndex;

            gridView.moveCurrentIndexLeft();
            gridView.selection = gridView.selectRange(gridView.previousIndex,
                                                      gridView.currentIndex);
            leftPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Right && shiftMultiSelection) {

            if(gridView.selection.length === 0) {
                rightPressed();
                initSelection(event);
                return;
            }

            if (gridView.previousIndex === -1)
                gridView.previousIndex = gridView.currentIndex;

            gridView.moveCurrentIndexRight();
            gridView.selection = gridView.selectRange(gridView.previousIndex,
                                                      gridView.currentIndex);
            rightPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Space) {
            spacePressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Escape) {
            escapePressed();
            gridView.selection = [];
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Delete) {
            deletePressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Enter || event.key === Qt.Key_Return) {
            enterPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Right){
            if(gridView.selection.length === 0) {
                rightPressed();
                initSelection(event);
                return;
            }
            gridView.moveCurrentIndexRight();
            gridView.selection = [docs[currentIndex]];
            gridView.previousIndex = gridView.currentIndex;
            rightPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Left){
            if(gridView.selection.length === 0) {
                leftPressed();
                initSelection(event);
                return;
            }
            gridView.moveCurrentIndexLeft();
            gridView.selection = [docs[currentIndex]];
            gridView.previousIndex = gridView.currentIndex;
            leftPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Up) {
            if(gridView.selection.length === 0) {
                upPressed();
                initSelection(event);
                return;
            }

            gridView.moveCurrentIndexUp();
            gridView.selection = [docs[currentIndex]];
            gridView.previousIndex = gridView.currentIndex;
            upPressed();
            event.accepted = true;
            return;
        }

        if (event.key === Qt.Key_Down) {
            if(gridView.selection.length === 0) {
                downPressed();
                initSelection(event);
                return;
            }

            gridView.moveCurrentIndexDown();
            gridView.selection = [docs[currentIndex]];
            gridView.previousIndex = gridView.currentIndex;
            downPressed();
            event.accepted = true;
            return;
        }

    } //key pressed
}


